package com.feasycom.feasybeacon.ui.about

import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import android.text.Spannable
import android.text.Spanned
import android.text.TextPaint
import android.text.style.UnderlineSpan
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.navigation.fragment.findNavController
import com.feasycom.feasybeacon.R
import com.feasycom.feasybeacon.databinding.FragmentAboutBinding
import com.feasycom.feasybeacon.ui.about.suota.activities.ScanActivity
import com.feasycom.feasybeacon.ui.base.BaseFragment

/**
 * 关于页面片段。
 */
class AboutFragment : BaseFragment<FragmentAboutBinding>() {

    override fun getViewBinding(
        inflater: LayoutInflater, container: ViewGroup?
    ): FragmentAboutBinding = FragmentAboutBinding.inflate(inflater, container, false)

    override fun initView() {
        initializeUI()
    }

    override fun initEvent() {
        initializeEvents()
    }

    override fun onResume() {
        super.onResume()
        binding.foot.navigationBar.check(R.id.about_nav)
    }

    /**
     * 初始化界面。
     */
    private fun initializeUI() {
        with(binding){
            // 设置导航栏
            foot.navigationBar.check(R.id.about_nav)
            // 设置标题和反馈按钮
            header.headerTitle.text = getString(R.string.about)
//            header.headerLeft.text = getString(R.string.suota_upgrade)
//            header.headerLeft.setOnClickListener { navigateToScanActivity() }
            header.headerRight.text = getString(R.string.feedback)
            header.headerRight.setOnClickListener { navigateToFeedbackActivity() }
            // 显示版本信息
            ver.text = getAppVersionName(requireContext())
        }

        // 设置按钮点击事件
        setupButtonListeners()
        // 移除下划线样式
        removeUnderlineSpans()
    }

    /**
     * 初始化事件监听器。
     */
    private fun initializeEvents() {
        with(binding){
            foot.beaconNav.setOnClickListener {
                navigateToFragment(R.id.beaconFragment)
            }
            foot.sensorNav.setOnClickListener {
                navigateToFragment(R.id.sensorFragment)
            }
            foot.settingNav.setOnClickListener {
                navigateToFragment(R.id.settingFragment)
            }
            foot.storeNav.setOnClickListener {
                navigateToFragment(R.id.storeFragment)
            }
        }

    }

    /**
     * 设置各个按钮的点击监听器。
     */
    private fun setupButtonListeners() {
        with(binding){
            wx.setOnClickListener { navigateToQRCodeActivity() }
            more.setOnClickListener { navigateToMoreActivity() }
            details.setOnClickListener { navigateToPdfActivity() }
            userAgreement.setOnClickListener { navigateToAgreementActivity(1) }
            privacyAgreement.setOnClickListener { navigateToAgreementActivity(2) }
        }
    }

    /**
     * 移除下划线样式。
     */
    private fun removeUnderlineSpans() {
        listOf(binding.phone, binding.email, binding.web).forEach { textView ->
            val spannableText = textView.text as? Spannable
            spannableText?.setSpan(NoUnderlineSpan(), 0, spannableText.length, Spanned.SPAN_MARK_MARK)
        }
    }

    /**
     * 获取应用版本名称。
     * @param context 上下文
     * @return 应用版本名称
     */
    private fun getAppVersionName(context: Context): String? {
        return try {
            val packageInfo = context.packageManager.getPackageInfo(context.packageName, 0)
            packageInfo.versionName
        } catch (e: PackageManager.NameNotFoundException) {
            e.printStackTrace()
            null
        }
    }

    override fun setMenuVisibility(menuVisible: Boolean) {
        super.setMenuVisibility(menuVisible)
        view?.visibility = if (menuVisible) View.VISIBLE else View.GONE
    }

    /**
     * 无下划线的 Span 类。
     */
    class NoUnderlineSpan : UnderlineSpan() {
        override fun updateDrawState(ds: TextPaint) {
            // 设置颜色并取消下划线
            ds.color = ds.linkColor
            ds.isUnderlineText = false
        }
    }

    /**
     * 导航到扫描活动。
     */
    private fun navigateToScanActivity() {
        startActivity(Intent(requireContext(), ScanActivity::class.java))
    }

    /**
     * 导航到反馈活动。
     */
    private fun navigateToFeedbackActivity() {
        startActivity(Intent(requireContext(), FeedbackActivity::class.java))
    }

    /**
     * 导航到二维码活动。
     */
    private fun navigateToQRCodeActivity() {
        startActivity(Intent(requireContext(), QRCodeActivity::class.java))
    }

    /**
     * 导航到更多活动。
     */
    private fun navigateToMoreActivity() {
        startActivity(Intent(requireContext(), AboutMoreActivity::class.java))
    }

    /**
     * 导航到 PDF 活动。
     */
    private fun navigateToPdfActivity() {
        startActivity(Intent(requireContext(), PdfActivity::class.java))
    }

    /**
     * 导航到协议活动。
     * @param type 协议类型（1: 用户协议, 2: 隐私政策）
     */
    private fun navigateToAgreementActivity(type: Int) {
        val intent = Intent(requireContext(), AgreementActivity::class.java).apply {
            putExtra("type", type)
        }
        startActivity(intent)
    }

    /**
     * 导航到指定的片段。
     * @param fragmentId 片段ID
     */
    private fun navigateToFragment(fragmentId: Int) {
        findNavController().navigate(fragmentId)
    }

}
